// 8n8 Automation Hub launcher for cPanel
// Starts n8n using the local node_modules binary.
// Works with cPanel "Application Manager" -> Node.js app.
//
// How it works:
// - Loads .env (if present).
// - Ensures n8n listens on the port cPanel provides (PORT), falling back to N8N_PORT or 5678.
// - Spawns the n8n process and inherits stdio so you can see logs in cPanel.

const path = require("path");
const { spawn } = require("child_process");
const fs = require("fs");

// Load .env if present
try {
  const dotenvPath = path.resolve(__dirname, ".env");
  if (fs.existsSync(dotenvPath)) {
    require("dotenv").config({ path: dotenvPath });
    console.log("[8n8] Loaded .env file");
  }
} catch (e) {
  console.warn("[8n8] .env not loaded:", e.message);
}

// cPanel often sets PORT. Make sure n8n uses it.
if (process.env.PORT && !process.env.N8N_PORT) {
  process.env.N8N_PORT = process.env.PORT;
}
if (!process.env.N8N_PORT) {
  process.env.N8N_PORT = "5678";
}

// Bind to all interfaces by default on cPanel
if (!process.env.N8N_HOST) {
  process.env.N8N_HOST = "0.0.0.0";
}

// Basic auth strongly recommended
if (!process.env.N8N_BASIC_AUTH_ACTIVE) process.env.N8N_BASIC_AUTH_ACTIVE = "true";
if (!process.env.N8N_BASIC_AUTH_USER) process.env.N8N_BASIC_AUTH_USER = "admin";
if (!process.env.N8N_BASIC_AUTH_PASSWORD) {
  console.warn("[8n8] N8N_BASIC_AUTH_PASSWORD not set. Please set a strong password in your .env!");
}

// Helpful defaults for reverse proxy / SSL setups
if (!process.env.N8N_PROTOCOL) process.env.N8N_PROTOCOL = "https";
if (!process.env.WEBHOOK_URL && process.env.PUBLIC_URL) {
  process.env.WEBHOOK_URL = process.env.PUBLIC_URL;
}

const n8nBin = path.resolve(__dirname, "node_modules", ".bin", process.platform === "win32" ? "n8n.cmd" : "n8n");
console.log("[8n8] Starting n8n from:", n8nBin);
console.log("[8n8] Host:", process.env.N8N_HOST, "Port:", process.env.N8N_PORT);

const child = spawn(n8nBin, ["start"], {
  stdio: "inherit",
  env: process.env,
});

child.on("exit", (code) => {
  console.log("[8n8] n8n exited with code", code);
  process.exit(code || 0);
});
